<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use App\Models\EventModel;
use App\Models\TicketModel;
use Hermawan\DataTables\DataTable;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Entities\User;

class EventController extends BaseController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "event";

    protected array $data = [
        "searchbar" => true,
        "statusbar" => false,
        "module"    => "Event Management",
    ];

    // Entities
    protected ?User $user = null;

    public function __construct()
    {
        helper(["form", "text", "input"]);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function index()
    {
        if (!$this->user->can('super.access', 'events.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        //$data = array_merge($this->data, $this->postList());
        $data = [];

        return view("{$this->dir}/post/list", $data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'events.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxList();
        }

        //$data = $this->postList();

        return view("{$this->dir}/post/list", $this->data);
    }

    protected function ajaxList()
    {
        $events = $this->getEventProvider();

        $events
            ->select(
                "events.id,
                events.event_code,
                events.event_name,
                events.event_description,
                events.active,
                users.username,
                users.first_name,
                users.last_name,
                users.middle_name,
                events.deleted_at"
            )
            ->join('users', 'events.created_by = users.id', 'left');

        if (!$this->user->can('super.access', 'admin.settings')) {
            $events->where("events.deleted_at", null);
        }

        $datatable = DataTable::of($events)
            ->filter(function ($events, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    $events
                        ->groupStart()
                        ->like('events.event_code', $search)
                        ->orLike('events.event_name', $search)
                        ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (isset($filter['active']) && $filter['active'] !== '') {
                        $events->where('events.active', $filter['active']);
                    }
                }
            })
            ->edit('id', function ($row) {
                return view("partials/elements/common/id", [
                    "id"          => $row->id,
                    "deleted"     => $row->deleted_at,
                ]);
            })
            ->edit('event_code', function ($row) {
                return "<strong>{$row->event_code}</strong>";
            })
            ->edit('event_description', function ($row) {
                return !empty($row->event_description) ? ellipsize($row->event_description, 30) : "None";
            })
            ->edit('active', function ($row) {
                return view("partials/elements/common/active_status", [
                    "active"    => $row->active
                ]);
            })
            ->add('creator', function ($row) {
                return view("partials/elements/common/table_user_name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "username"    => $row->username,
                ]);
            })
            ->add('action', function ($row) {
                return view("partials/elements/common/action_buttons", [
                    "id"        => $row->id,
                    "active"    => $row->active,
                    "editUrl"   => event_url("edit/{$row->id}"),
                ]);
            })
            ->toJson(true);

        return $datatable;
    }

    public function createView()
    {
        if (!$this->user->can('super.access', 'events.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $this->data["searchbar"] = false;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/create", $this->data);
        }

        return view("{$this->dir}/post/create", $this->data);
    }

    public function createAction()
    {
        if (!$this->user->can('super.access', 'events.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxCreate();
        }

        //return $this->postCreate();
    }

    protected function ajaxCreate()
    {
        $events = $this->getEventProvider();

        $data = clean_input_data($this->request->getPost());

        try {
            $data['created_by'] = $this->user->id;
        } catch (\ErrorException $ex) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "You are not logged in. Please logged in first.",
            ]);
        }

        if (!$events->save($data)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $events->errors(),
                "data"      => null,
            ]);
        }

        session()->setFlashdata("success", "Event has been successfully created.");

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Event has been successfully created.",
        ]);
    }

    protected function postCreate()
    {
        $events = $this->getEventProvider();

        $data = $this->request->getPost();
        $data['created_by'] = $this->user->id;

        if (!$events->save($data)) {
            return redirect()->back()->withInput()->with('error', $events->errors());
        }

        return redirect()->to('branch')->with('success', 'Event has been successfully added!');
    }

    public function editView($id)
    {
        if (!$this->user->can('super.access', 'events.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $events = $this->getEventProvider();

        $event = $events->find($id);

        if ($event === null) {
            return redirect()->back()->with("error", "There is a problem fetching event data. Event cannot be edit right now.");
        }

        $this->data["searchbar"] = false;
        $this->data["event"] = $event;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/edit", $this->data);
        }

        return view("{$this->dir}/post/edit", $this->data);
    }

    public function editAction($id = null)
    {
        if (!$this->user->can('super.access', 'events.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxEdit($id);
        }

        //return $this->postEdit();
    }

    protected function ajaxEdit($id = null)
    {
        $events         = $this->getEventProvider();
        $data           = clean_input_data($this->request->getPost());
        $data['id']     = $data['id'] ?? $id;
        $data['active'] = isset($data['active']);

        unset($data['event_code']);

        if (!$events->save($data)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $events->errors(),
                "data"      => null,
            ]);
        }

        session()->setFlashdata("success", "Event has been successfully updated.");

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Event has been successfully created.",
        ]);
    }

    public function activate($id = null)
    {
        if (!$this->user->can('super.access', 'events.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        $events     = $this->getEventProvider();

        if ($events->activate($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Event(s) Activated",
                "data"      => "Event(s) has been activated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Activation Failed",
                "data"      => "Event(s) cannot be activated."
            ]);
        }
    }

    public function deactivate($id = null)
    {
        if (!$this->user->can('super.access', 'events.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        $events     = $this->getEventProvider();

        if ($events->deactivate($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Event(s) Deactivated",
                "data"      => "Event(s) has been deactivated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Deactivation Failed",
                "data"      => "Event(s) cannot be deactivated."
            ]);
        }
    }

    public function delete($id = null)
    {
        if (!$this->user->can('super.access', 'events.delete')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        $events     = $this->getEventProvider();
        $purge      = $this->user->can('super.access', 'admin.settings');

        if ($events->deleteMultiple($id, $purge)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Event(s) Deleted",
                "data"      => "Event(s) has been deleted."
            ]);
        }

        return $this->response->setJSON([
            "success"   => false,
            "title"     => "Deletion Failed",
            "data"      => "Event(s) cannot be deleted."
        ]);
    }

    protected function getEventProvider(): EventModel
    {
        return new EventModel();
    }

    protected function getTicketProvider(): TicketModel
    {
        return new TicketModel();
    }
}
