<?php

namespace App\Controllers\Api;

use App\Models\UserModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Controllers\RegisterController as ShieldRegisterController;
use CodeIgniter\Shield\Entities\User;
use CodeIgniter\Shield\Exceptions\ValidationException;
use CodeIgniter\Shield\Validation\ValidationRules;
use CodeIgniter\Shield\Authorization\AuthorizationException;
use Config\Database;
use Hermawan\DataTables\DataTable;

class UserController extends ShieldRegisterController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "user";

    protected array $data = [
        "searchbar" => true,
        "statusbar" => false,
        "module"    => "User Management"
    ];

    // Entities
    protected ?User $user = null;

    public function __construct()
    {
        helper(["form", "input"]);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function index()
    {
        if (!$this->user->can('super.access', 'users.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        $data = array_merge($this->data, $this->postList());

        return view("{$this->dir}/post/list", $data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'users.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxList();
        }

        $data = array_merge($this->data, $this->postList());

        return view("{$this->dir}/post/list", $data);
    }

    protected function ajaxList()
    {
        $users = $this->getUserProvider();

        $users
            ->select(
                "users.id,
                users.username,
                identities.secret AS email,
                branches.branch_name,
                users.first_name,
                users.last_name,
                users.middle_name,
                users.phone_number,
                groups.group,
                users.status,
                users.last_active,
                users.created_at,
                users.created_by"
            )
            ->join("auth_identities AS identities", "identities.user_id = users.id", "left")
            ->join("branches", "users.branch_id = branches.id", "left")
            ->join("auth_groups_users AS groups", "groups.user_id = users.id", "left")
            ->where("identities.type", "email_password")
            ->where("users.deleted_at", null)
            ->where("users.id <>", $this->user->id);

        // if ($this->user) {
        //     $users->where("users.id <>", $this->user->id);
        // } else {
        //     $users->where("users.id", null);
        // }

        $datatable = DataTable::of($users)
            ->filter(function ($users, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    $users
                        ->groupStart()
                        ->like("users.username", $search)
                        ->orLike("identities.secret", $search)
                        ->orLike("users.first_name", $search)
                        ->orLike("users.last_name", $search)
                        ->orLike("users.phone_number", $search)
                        ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (!empty($filter["branch"])) {
                        $users->where("branches.id", $filter["branch"]);
                    }
                    if (!empty($filter["group"])) {
                        $users->where("groups.group", $filter["group"]);
                    }
                    if (!empty($filter["status"])) {
                        switch ($filter["status"]) {
                            case '1':
                                $users->where("users.status", "banned");
                                break;

                            case '2':
                                $users->where("users.status", null);
                                break;
                        }
                    }
                }
            })
            ->format("branch_name", function ($value, $meta) {
                return $value ?? "None";
            })
            ->format("group", function ($value, $meta) {
                return ucwords($value);
            })
            ->format("status", function ($value, $meta) {
                return view("partials/elements/users/status", [
                    "status"    => $value
                ]);
            })
            ->format("last_active", function ($value, $meta) {
                return $value ? date("Y-M-d | h:i A", strtotime($value)) : "Never";
            })
            ->format("created_at", function ($value, $meta) {
                return date("Y-M-d | h:i A", strtotime($value));
            })
            ->add("name", function ($row) {
                return view("partials/elements/common/table_user_name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "username"    => $row->username,
                    "email"       => $row->email,
                ]);
            })
            ->add("action", function ($row) {
                return view("partials/elements/users/action_buttons", [
                    "id"        => $row->id,
                    "active"    => $row->status !== 'banned',
                    "editUrl"   => user_url("edit/{$row->id}"),
                ]);
            })
            ->toJson(true);

        return $datatable;
    }

    protected function postList()
    {
        if ($this->request->is("post")) {

            $search = $this->request->getVar("search");

            $builder = Database::connect()->table("users")->where("users.deleted_at IS NULL");

            $builder->select(
                "users.id,
                users.username,
                auth_identities.secret AS email,
                users.first_name,
                users.last_name,
                users.middle_name,
                branches.branch_name,
                auth_groups_users.group,
                users.active"
            );

            $builder
                ->join("auth_identities", "auth_identities.user_id = users.id", "left")
                ->join("branches", "profiles.branch_id = branches.id", "left")
                ->join("auth_groups_users", "auth_groups_users.user_id = users.id", "left");

            if ($search) {
                $builder->groupStart()
                    ->like("users.username", $search)
                    ->orLike("auth_identities.secret", $search)
                    ->orLike("users.first_name", $search)
                    ->orLike("users.last_name", $search)
                    ->groupEnd();
            }

            $users = $builder->get()->getResult();

            $data = [
                "users"     => $users,
                "search"    => $search,
            ];

            return $data;
        }
    }

    public function createView()
    {
        if (!$this->user->can('super.access', 'users.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $this->data["searchbar"] = false;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/create", $this->data);
        }

        return view("{$this->dir}/post/create", $this->data);
    }

    public function createAction()
    {
        if (!$this->user->can('super.access', 'users.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxCreate();
        }

        return $this->postCreate();
    }

    protected function ajaxCreate()
    {
        if ($this->request->isAJAX()) {

            $users      = $this->getUserProvider();
            $userRules  = $this->getUserValidationRules();
            $groupRules = $this->getGroupValidationRules();

            // Merge all rules
            $allRules = array_merge($userRules, $groupRules);

            if (!$this->validateData($this->request->getPost(), $allRules, [], setting("Auth.DBGroup"))) {
                return $this->response->setJSON([
                    "success"   => false,
                    "errors"    => $this->validator->getErrors(),
                    "data"      => null,
                ]);
            }

            $userAllowedPostFields = array_keys($userRules);
            $user = $this->getUserEntity();
            $capitalize = ['first_name', 'middle_name', 'last_name'];
            $data = clean_input_data($this->request->getPost($userAllowedPostFields), $capitalize);

            $data['active'] = true;

            try {
                $data['created_by'] = $this->user->id;
            } catch (\ErrorException $ex) {
                return $this->response->setJSON([
                    "success"   => false,
                    "errors"    => null,
                    "data"      => "You are not logged in. Please logged in first.",
                ]);
            }

            $user->fill($data);

            // Workaround for email only registration/login
            if ($user->username === null) {
                $user->username = null;
            }

            // Save the user
            try {
                $users->save($user);
            } catch (ValidationException $e) {
                return $this->response->setJSON([
                    "success"   => false,
                    "errors"    => null,
                    "data"      => $users->errors(),
                ]);
            }

            // To get the complete user object with ID, we need to get from the database
            $user = $users->findById($users->getInsertID());

            // Add to group
            $groupAllowedPostFields = array_key_first($groupRules);
            $group = $this->request->getVar($groupAllowedPostFields);

            try {
                $user->syncGroups($group);
            } catch (AuthorizationException $e) {
                return $this->response->setJSON([
                    "success"   => false,
                    "errors"    => null,
                    "data"      => $e->getMessage(),
                ]);
            }

            session()->setFlashdata("success", "User has been successfully created.");

            return $this->response->setJSON([
                "success"   => true,
                "errors"    => null,
                "data"      => "User has been successfully created.",
            ]);
        } else {

            return $this->response->setStatusCode(400)->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "Invalid request."
            ]);
        }
    }

    protected function postCreate()
    {
        $users      = $this->getUserProvider();
        $userRules  = $this->getUserValidationRules();
        $groupRules = $this->getGroupValidationRules();

        // Merge all rules
        $rules = array_merge($userRules, $groupRules);

        if (!$this->validateData($this->request->getPost(), $rules, [], setting("Auth.DBGroup"))) {
            return redirect()->back()->withInput()->with("errors", $this->validator->getErrors());
        }

        // Save the user
        $userAllowedPostFields = array_keys($userRules);
        $user = $this->getUserEntity();
        $user->fill($this->request->getPost($userAllowedPostFields));

        // Workaround for email only registration/login
        if ($user->username === null) {
            $user->username = null;
        }

        try {
            $users->save($user);
        } catch (ValidationException $e) {
            return redirect()->back()->withInput()->with("errors", ["main" => $users->errors()]);
        }

        // To get the complete user object with ID, we need to get from the database
        $user = $users->findById($users->getInsertID());

        // Add to group
        $groupAllowedPostFields = array_keys($groupRules);
        $group = (string) $this->request->getPost($groupAllowedPostFields);

        try {
            $user->addGroup($group);
        } catch (AuthorizationException $e) {
            return redirect()->back()->withInput()->with("errors", ["main" => $e->getMessage()]);
        }

        // Set the user active
        $user->activate();

        return redirect()->to("user")->with("success", "User successfully created.");
    }

    public function editView($id)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $users          = $this->getUserProvider();
        $user           = $users->findById($id);
        $groups         = $user->getGroups();
        $user->group    = $groups ? $groups[0] : null;

        if ($user === null) {
            return redirect()->back()->with("error", "There is a problem fetching user\" data. User cannot be edit right now.");
        }

        $this->data["searchbar"] = false;
        $this->data["user"] = $user;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/edit", $this->data);
        }

        return view("{$this->dir}/post/edit", $this->data);
    }

    public function editAction($id = null)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxEdit($id);
        }

        //return $this->postEdit();
    }

    protected function ajaxEdit($id = null)
    {
        $users          = $this->getUserProvider();
        $oldData        = $users->findById($id);

        // Validate here first, since some things,
        // like the password, can only be validated properly here.
        $userRules      = $this->getUserValidationRules();
        $userIdRules    = $this->getUserIdValidationRules();
        $usernameRules  = $this->getUsernameValidationRules();
        $emailRules     = $this->getEmailValidationRules();
        $groupRules     = $this->getGroupValidationRules();
        $passwordRules  = $this->getPasswordValidationRules();

        // Merge all rules
        $allRules       = array_merge($userIdRules, $userRules, $usernameRules, $emailRules, $groupRules, $passwordRules);
        $data           = $this->request->getPost();
        $identifier     = array_key_first($userIdRules);
        $data[$identifier] = $data[$identifier] ?? $id;

        if (!$this->validateData($data, $allRules, [], setting("Auth.DBGroup"))) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $this->validator->getErrors(),
                "data"      => null,
            ]);
        }

        // Get user rules fields with corresponding user field value
        $userUpdateRules = array_filter(array_merge($userIdRules, $userRules), function ($key) use ($data) {
            return !empty($data[$key]);
        }, ARRAY_FILTER_USE_KEY);

        $userAllowedPostFields  = array_keys($userUpdateRules);
        $user                   = $this->getUserEntity();
        $capitalize             = ['first_name', 'middle_name', 'last_name'];
        $data                   = clean_input_data($this->request->getPost($userAllowedPostFields), $capitalize);
        // $data['status']         = $this->request->getPost('status') !== null;

        $user->fill($data);

        // Save the user
        try {
            $users->save($user);
        } catch (ValidationException $e) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => $users->errors(),
            ]);
        }

        if ($this->request->getPost('suspend') !== null) {
            $user->ban();
        }

        // Add to group
        $groupAllowedPostFields = array_key_first($groupRules);
        $group = (string) $this->request->getPost($groupAllowedPostFields);

        // Check if changing group first to not over populate logs on update
        if (!in_array($group, $oldData->getGroups())) {
            try {
                $user->syncGroups($group);
            } catch (AuthorizationException $e) {
                return $this->response->setJSON([
                    "success"   => false,
                    "errors"    => null,
                    "data"      => $e->getMessage(),
                ]);
            }
        }

        session()->setFlashdata("success", "User has been successfully updated.");

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "User has been successfully updated.",
        ]);
    }


    protected function postEdit($id)
    {
        $users          = $this->getUserProvider();
        $oldData        = $users->findById($id);

        // Validate here first, since some things,
        // like the password, can only be validated properly here.
        $userRules      = $this->getUserValidationRules();
        $userIdRules    = $this->getUserIdValidationRules();
        $usernameRules  = $this->getUsernameValidationRules();
        $emailRules     = $this->getEmailValidationRules();
        $groupRules     = $this->getGroupValidationRules();
        $passwordRules  = $this->getPasswordValidationRules();

        // Merge all rules
        $allRules = array_merge($userIdRules, $userRules, $usernameRules, $emailRules, $groupRules, $passwordRules);

        $data = $this->request->getPost();

        $identifier = array_key_first($userIdRules);

        $data[$identifier] = $data[$identifier] ?? $id;

        if (!$this->validateData($data, $allRules, [], setting("Auth.DBGroup"))) {
            return redirect()->back()->withInput()->with("errors", $this->validator->getErrors());
        }

        // Get user rules fields with corresponding user field value
        $userUpdateRules = array_filter(array_merge($userIdRules, $userRules), function ($key) use ($data) {
            return !empty($data[$key]);
        }, ARRAY_FILTER_USE_KEY);

        $userAllowedPostFields  = array_keys($userUpdateRules);
        $user                   = $this->getUserEntity();
        $data                   = clean_input_data($this->request->getPost($userAllowedPostFields));
        $data['active']         = $this->request->getPost('active') !== null;

        $user->fill($data);

        // Save the user
        try {
            $users->save($user);
        } catch (ValidationException $e) {
            return redirect()->back()->withInput()->with("data", $users->errors());
        }

        // Add to group
        $groupAllowedPostFields = array_key_first($groupRules);
        $group = (string) $this->request->getPost($groupAllowedPostFields);

        // Check if changing group first to not over populate logs on update
        if (!in_array($group, $oldData->getGroups())) {
            try {
                $user->syncGroups($group);
            } catch (AuthorizationException $e) {
                return redirect()->back()->withInput()->with("data", $e->getMessage());
            }
        }

        return redirect()->to(user_url())->with("success", "User has been successfully updated.");
    }

    public function activate($id = null)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        /** @var \App\Models\UserModel $users */
        $users      = $this->getUserProvider();

        if ($users->activateMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "User(s) Activated",
                "data"      => "User(s) has been activated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Activation Failed",
                "data"      => "User(s) cannot be found. Activation is terminated."
            ]);
        }
    }

    public function deactivate($id = null)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        /** @var \App\Models\UserModel $users */
        $users      = $this->getUserProvider();

        if ($users->deactivateMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "User(s) Deactivated",
                "data"      => "User(s) has been deactivated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Deactivation Failed",
                "data"      => "User(s) cannot be found. Deactivation is terminated."
            ]);
        }
    }

    public function ban($id = null)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        /** @var \App\Models\UserModel $users */
        $users      = $this->getUserProvider();

        if ($users->banMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "User(s) Suspended",
                "data"      => "User(s) has been suspended."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Suspension Failed",
                "data"      => "User(s) cannot be found. Suspension is terminated."
            ]);
        }
    }

    public function unban($id = null)
    {
        if (!$this->user->can('super.access', 'users.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        /** @var \App\Models\UserModel $users */
        $users      = $this->getUserProvider();

        if ($users->unbanMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "User(s) Reactivated",
                "data"      => "User(s) has been reactivated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Reactivation Failed",
                "data"      => "User(s) cannot be found. Reactivation is terminated."
            ]);
        }
    }

    public function delete($id = null)
    {
        if (!$this->user->can('super.access', 'users.delete')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        /** @var \App\Models\UserModel $users */
        $users      = $this->getUserProvider();

        if ($users->deleteMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "User(s) Deleted",
                "data"      => "User(s) has been deleted."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Deletion Failed",
                "data"      => "User(s) cannot be deleted."
            ]);
        }
    }

    /**
     * Returns the rules that should be used for user validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getUserValidationRules(): array
    {
        $validation = new ValidationRules();

        $rules = $validation->getRegistrationRules();

        return $rules;
    }

    /**
     * Returns the rules for updating user that should be used for user id validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getUserIdValidationRules(): array
    {
        $rules = setting("Validation.userIdValidation");

        return $rules;
    }

    /**
     * Returns the rules for updating user that should be used for username validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getUsernameValidationRules(): array
    {
        $rules = setting("Validation.usernameValidation");

        return $rules;
    }

    /**
     * Returns the rules for updating user that should be used for email validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getEmailValidationRules(): array
    {
        $rules = setting("Validation.emailValidation");

        return $rules;
    }

    /**
     * Returns the rules that should be used for group validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getGroupValidationRules(): array
    {
        $rules = setting("Validation.groupValidation");

        return $rules;
    }

    /**
     * Returns the rules for updating user that should be used for password validation.
     *
     * @return array<string, array<string, list<string>|string>>
     */
    protected function getPasswordValidationRules(): array
    {
        $rules = setting("Validation.passwordValidation");

        return $rules;
    }
}
