<?php

namespace App\Validation;

use Config\Database;

/**
 * Custom Validation Rules.
 *
 * @see \App\Validation\CustomRulesTest
 */
class CustomRules
{
    /**
     * Checks the database to see if the given combined field value is unique. Can
     * ignore a single record by field/value to make it useful during
     * record updates.
     *
     * Example:
     *    is_unique_combination[table.field1,field2,value2,ignore_field,ignore_value]
     *    is_unique_combination[tickets.id_number,event_id,1,id,1]
     *
     * @param array|bool|float|int|object|string|null $str
     */
    public function is_unique_combination(string $str, string $field, array $data, ?string &$error = null): bool
    {
        if (is_object($str) || is_array($str)) {
            return false;
        }

        [$field1, $field2, $value2, $ignoreField, $ignoreValue] = array_pad(
            explode(',', $field),
            5,
            null
        );

        sscanf($field1, '%[^.].%[^.]', $table, $field1);

        $row = Database::connect($data['DBGroup'] ?? null)
            ->table($table)
            ->select('1')
            ->where($field1, $str)
            ->limit(1);

        if (
            $field2 !== null && $field2 !== ''
            && $value2 !== null && $value2 !== ''
            && !preg_match('/^\{(\w+)\}$/', $value2)
        ) {
            $row = $row->where($field2, $value2);
        }

        if (
            $ignoreField !== null && $ignoreField !== ''
            && $ignoreValue !== null && $ignoreValue !== ''
            && !preg_match('/^\{(\w+)\}$/', $ignoreValue)
        ) {
            $row = $row->where("{$ignoreField} !=", $ignoreValue);
        }

        return $row->get()->getRow() === null;
    }

    /**
     * Checks the database to see if the given combined field value is unique. Can
     * ignore a single record by field/value to make it useful during
     * record updates.
     *
     * Example:
     *    is_unique_combination[table.field1,field2,value2,field3,value3,ignore_field,ignore_value]
     *    is_unique_combination[tickets.id_number,event_id,1,branch_id,1,id,1]
     *
     * @param array|bool|float|int|object|string|null $str
     */
    public function is_unique_combinations(string $str, string $field, array $data, ?string &$error = null): bool
    {
        if (is_object($str) || is_array($str)) {
            return false;
        }

        [$field1, $field2, $value2, $field3, $value3, $ignoreField, $ignoreValue] = array_pad(
            explode(',', $field),
            7,
            null
        );

        sscanf($field1, '%[^.].%[^.]', $table, $field1);

        $row = Database::connect($data['DBGroup'] ?? null)
            ->table($table)
            ->select('1')
            ->where($field1, $str)
            ->limit(1);

        if (
            $field2 !== null && $field2 !== ''
            && $value2 !== null && $value2 !== ''
            && !preg_match('/^\{(\w+)\}$/', $value2)
        ) {
            $row = $row->where($field2, $value2);
        }

        if (
            $field3 !== null && $field3 !== ''
            && $value3 !== null && $value3 !== ''
            && !preg_match('/^\{(\w+)\}$/', $value3)
        ) {
            $row = $row->where($field3, $value3);
        }

        if (
            $ignoreField !== null && $ignoreField !== ''
            && $ignoreValue !== null && $ignoreValue !== ''
            && !preg_match('/^\{(\w+)\}$/', $ignoreValue)
        ) {
            $row = $row->where("{$ignoreField} !=", $ignoreValue);
        }

        return $row->get()->getRow() === null;
    }

    /**
     * Checks if user correctly enters their old password
     */
    public function old_password(string $password = null): bool
    {
        return auth()->check([
            'email'    => auth()->user()->email,
            'password' => $password,
        ])->isOK();
    }
}
