<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;
use App\Models\BranchModel;
use Hermawan\DataTables\DataTable;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Entities\User;

class BranchController extends BaseController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "branch";

    protected array $data = [
        "searchbar" => true,
        "statusbar" => false,
        "module"    => "Branch Management",
    ];

    // Entities
    protected ?User $user = null;

    public function __construct()
    {
        helper(["form", "input", "array"]); // $fizz = dot_array_search('foo.buzz.fizz', $data);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function index()
    {
        if (!$this->user->can('super.access', 'branches.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        //$data = array_merge($this->data, $this->postList());
        $data = [];

        return view("{$this->dir}/post/list", $data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'branches.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxList();
        }

        //$data = $this->postList();

        return view("{$this->dir}/post/list", $this->data);
    }

    protected function ajaxList()
    {
        $branches = $this->getBranchProvider();

        $branches
            ->select(
                "branches.id,
                branches.branch_code,
                branches.branch_name,
                branches.active,
                branches.deleted_at,
                users.username,
                users.first_name,
                users.last_name,
                users.middle_name"
            )
            ->join('users', 'branches.created_by = users.id', 'left');

        if (!$this->user->can('super.access', 'admin.settings')) {
            $branches->where("branches.deleted_at", null);
        }

        $datatable = DataTable::of($branches)
            ->filter(function ($branches, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    $branches
                        ->groupStart()
                        ->like('branches.branch_code', $search)
                        ->orLike('branches.branch_name', $search)
                        ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (isset($filter['active']) && $filter['active'] !== '') {
                        $branches->where('branches.active', $filter['active']);
                    }
                }
            })
            ->edit('id', function ($row) {
                return view("partials/elements/common/id", [
                    "id"          => $row->id,
                    "deleted"     => $row->deleted_at,
                ]);
            })
            ->edit('branch_code', function ($row) {
                return "<strong>{$row->branch_code}</strong>";
            })
            ->edit('active', function ($row) {
                return view("partials/elements/common/active_status", [
                    "active"    => $row->active
                ]);
            })
            ->add('creator', function ($row) {
                return view("partials/elements/common/table_user_name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "username"    => $row->username,
                ]);
            })
            ->add('action', function ($row) {
                return view("partials/elements/common/action_buttons", [
                    "id"        => $row->id,
                    "active"    => $row->active,
                    "editUrl"   => branch_url("edit/{$row->id}"),
                    "deleted"   => $row->deleted_at,
                ]);
            })
            ->toJson(true);

        return $datatable;
    }

    public function createView()
    {
        if (!$this->user->can('super.access', 'branches.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $this->data["searchbar"] = false;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/create", $this->data);
        }

        return view("{$this->dir}/post/create", $this->data);
    }

    public function createAction()
    {
        if (!$this->user->can('super.access', 'branches.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxCreate();
        }

        //return $this->postCreate();
    }

    protected function ajaxCreate()
    {
        $branches = $this->getBranchProvider();

        $data = $this->request->getPost();

        try {
            $data['created_by'] = $this->user->id;
        } catch (\ErrorException $ex) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "You are not logged in. Please logged in first.",
            ]);
        }

        if (!$branches->save($data)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $branches->errors(),
                "data"      => null,
            ]);
        }

        session()->setFlashdata("success", "Branch has been successfully created.");

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Branch has been successfully created.",
        ]);
    }

    protected function postCreate()
    {
        $branches = $this->getBranchProvider();

        $data = $this->request->getPost();
        $data['created_by'] = $this->user->id;

        if (!$branches->save($data)) {
            return redirect()->back()->withInput()->with('error', $branches->errors());
        }

        return redirect()->to('branch')->with('success', 'Branch has been successfully added!');
    }

    public function editView($id)
    {
        if (!$this->user->can('super.access', 'branches.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $branches = $this->getBranchProvider();

        $branch = $branches->find($id);

        if ($branch === null) {
            return redirect()->back()->with("error", "There is a problem fetching branch data. Branch cannot be edit right now.");
        }

        $this->data["searchbar"] = false;
        $this->data["branch"] = $branch;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/edit", $this->data);
        }

        return view("{$this->dir}/post/edit", $this->data);
    }

    public function editAction($id = null)
    {
        if (!$this->user->can('super.access', 'branches.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxEdit($id);
        }

        //return $this->postEdit();
    }

    protected function ajaxEdit($id = null)
    {
        $branches = $this->getBranchProvider();

        $data           = $this->request->getPost();
        $data['id']     = $data['id'] ?? $id;
        $data['active'] = isset($data['active']);

        if (!$branches->save($data)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $branches->errors(),
                "data"      => null,
            ]);
        }

        session()->setFlashdata("success", "Branch has been successfully updated.");

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Branch has been successfully created.",
        ]);
    }

    public function activate($_id = null)
    {
        if (!$this->user->can('super.access', 'branches.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $_id ?? 0;
        $branches   = $this->getBranchProvider();

        if ($branches->activateMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Branch(es) Activated",
                "data"      => "Branch(es) has been activated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Activation Failed",
                "data"      => "Branch(es) cannot be activated."
            ]);
        }
    }

    public function deactivate($id = null)
    {
        if (!$this->user->can('super.access', 'branches.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $_id ?? 0;
        $branches   = $this->getBranchProvider();

        if ($branches->deactivateMultiple($id)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Branch(es) Deactivated",
                "data"      => "Branch(es) has been deactivated."
            ]);
        } else {
            return $this->response->setJSON([
                "success"   => false,
                "title"     => "Deactivation Failed",
                "data"      => "Branch(es) cannot be deactivated."
            ]);
        }
    }

    public function delete($id = null)
    {
        if (!$this->user->can('super.access', 'branches.delete')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $_id ?? 0;
        $branches   = $this->getBranchProvider();
        $purge      = $this->user->can('super.access', 'admin.settings');

        if ($branches->deleteMultiple($id, $purge)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Branch(es) Deleted",
                "data"      => "Branch(es) has been deleted."
            ]);
        }

        return $this->response->setJSON([
            "success"   => false,
            "title"     => "Deletion Failed",
            "data"      => "Branch(es) cannot be deleted."
        ]);
    }

    protected function getBranchProvider()
    {
        return new BranchModel();
    }
}
