<?php

namespace App\Controllers\Api;

use App\Controllers\Api\UserController;
use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Exceptions\ValidationException;
use App\Models\LogModel;
use App\Models\UserModel;
use Config\RiceData;
use Hermawan\DataTables\DataTable;

class ProfileController extends BaseController
{
    protected bool $isAjaxProcess   = true;
    protected string $dir           = "profile";
    protected array $data           = [
        "searchbar"     => false,
        "statusbar"     => true,
        "module"        => "My Account",
        "showCreator"   => true
    ];

    // Config
    protected ?RiceData $rices      = null;

    public function __construct()
    {
        helper(["form", "input", "log"]);

        if (!auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function viewProfile()
    {
        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/profile", $this->data);
        }

        return view("{$this->dir}/post/profile", $this->data);
    }

    public function viewActivity()
    {
        if ($this->isAjaxProcess) {
            return $this->ajaxLog();
        }

        return view("{$this->dir}/post/profile", $this->data);
    }

    protected function ajaxLog()
    {
        $logs = $this->getLogProvider();

        $logs->select("id, timestamp, action, entity_id, module, performed_by");

        $datatable = DataTable::of($logs)
            ->format('timestamp', function ($value, $meta) {
                return date("Y-M-d | h:i A", strtotime($value));
            })
            ->add('description', function ($row) {
                return log_descriptor($row);
            })
            ->toJson(true);

        return $datatable;
    }

    public function viewSettings()
    {
        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/settings", $this->data);
        }

        return view("{$this->dir}/post/settings", $this->data);
    }

    public function updateProfile()
    {
        $users          = $this->getUserProvider();
        $data           = clean_input_data($this->request->getPost());
        $rules          = $this->getProfileValidation();
        $user           = auth()->user();

        if ($user === null) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "You are not logged in. Please log in first.",
            ]);
        }

        $data['id']     = $user->id;

        if (!$this->validateData($data, $rules)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $this->validator->getErrors(),
                "data"      => null,
            ]);
        }

        $user->fill($data);

        try {
            $users->save($user);
        } catch (ValidationException $e) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => $users->errors(),
            ]);
        }

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Profile Updated!",
        ]);
    }

    public function changePassword()
    {
        $users          = $this->getUserProvider();
        $data           = clean_input_data($this->request->getPost());
        $rules          = $this->getPasswordValidation();
        $user           = auth()->user();

        if ($user === null) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "You are not logged in. Please log in first.",
            ]);
        }

        if (!$this->validateData($data, $rules)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $this->validator->getErrors(),
                "data"      => null,
            ]);
        }

        $user->fill($data);

        try {
            $users->save($user);
        } catch (ValidationException $e) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => $users->errors(),
            ]);
        }

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Password Changed!",
        ]);
    }

    protected function getProfileValidation()
    {
        /** @var \Config\Validation $config */
        $config = config('Validation');

        $rules = $config->registration;
        $rules['id']['rules'] = ['permit_empty'];
        $rules['username']['rules'] = [
            'required',
            'max_length[30]',
            'min_length[3]',
            'alpha_numeric',
            'is_unique[users.username,id,{id}]',
        ];
        $rules['email']['rules'] = [
            'required',
            'max_length[254]',
            'valid_email',
            'is_unique[auth_identities.secret,user_id,{id}]',
        ];
        unset($rules['password'], $rules['password_confirm']);

        return $rules;
    }

    protected function getPasswordValidation()
    {
        $rules = [
            'old_password' => [
                'label' => 'Old Password',
                'rules' => [
                    'required',
                    'old_password',
                ],
                'errors' => [
                    'required'      => '{field} is required.',
                    'old_password'  => 'Old password is incorrect.',
                ]
            ],
            'password' => [
                'label' => 'Auth.password',
                'rules' => [
                    'required',
                    'max_byte[72]',
                    'strong_password[]',
                ],
                'errors' => [
                    'required'      => '{field} is required.',
                    'max_byte'      => 'Auth.errorPasswordTooLongBytes',
                ]
            ],
            'password_confirm' => [
                'label' => 'Auth.passwordConfirm',
                'rules' => [
                    'required',
                    'matches[password]',
                ],
                'errors'   => [
                    'required'      => 'Please confirm password.',
                    'matches'       => 'Password did not match or is empty.',
                ],
            ],
        ];

        return $rules;
    }

    protected function getLogProvider(): LogModel
    {
        return new LogModel();
    }

    protected function getUserProvider(): UserModel
    {
        return new UserModel();
    }
}
