<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;
use App\Models\TicketModel;
use App\Models\EventModel;
use App\Models\BranchModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Entities\User;
use Config\RiceData;
use Hermawan\DataTables\DataTable;
use \Mpdf\Mpdf;

class ReportController extends BaseController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "report";

    protected array $data = [
        "searchbar" => false,
        "statusbar" => false,
        "module"    => "Ticket Reports"
    ];

    // Models
    protected ?BranchModel $branches = null;
    protected ?TicketModel $tickets = null;
    protected ?EventModel $events   = null;

    // Entities
    protected ?User $user           = null;

    // Config
    protected ?RiceData $rices      = null;

    public function __construct()
    {
        helper(["form", "number", "array"]); // $fizz = dot_array_search('foo.buzz.fizz', $data);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function list()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        return view("{$this->dir}/post/list", $this->data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->generate();
        }

        //$data = $this->postList();

        return view("{$this->dir}/post/list", $this->data);
    }

    public function generate()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $filter     = $this->request->getPost();
        $tickets    = $this->getTicketProvider();
        $title      = 'List of Tickets';

        $tickets
            ->select(
                "tickets.id,
                events.event_code,
                tickets.ticket_number,
                branches.branch_code,
                branches.branch_name,
                tickets.id_number,
                tickets.first_name,
                tickets.last_name,
                tickets.middle_name,
                tickets.email,
                tickets.phone_number,
                tickets.position,
                tickets.department,
                tickets.year,
                tickets.track,
                tickets.req_submitted,
                tickets.date_attend,
                tickets.date_leave,
                tickets.created_at,
                encoder.username AS encoder_username,
                encoder.first_name AS encoder_first_name,
                encoder.last_name AS encoder_last_name,
                encoder.middle_name AS encoder_middle_name"
            )
            ->join('users AS encoder', 'tickets.created_by = encoder.id', 'left')
            ->join('events', 'tickets.event_id = events.id', 'inner')
            ->join('branches', 'tickets.branch_id = branches.id', 'left')
            ->where("tickets.deleted_at", null);

        if (!$this->user->can('tickets.view-all')) {
            $tickets->where('branches.id', $this->user->branch_id);
        }

        if (isset($filter)) {
            if (!empty($filter["event"])) {
                $tickets->where("events.id", $filter["event"]);
            } else {
                $tickets->where("events.active", true);
            }
            if (!empty($filter["branch"])) {
                $tickets->where("branches.id", $filter["branch"]);
            }
            if (!empty($filter["position"])) {
                $tickets->where("tickets.position", $filter["position"]);
            }
            if (!empty($filter["department"])) {
                $tickets->where("tickets.department", $filter["department"]);
            }
            if (isset($filter["requirements"]) && $filter["requirements"] !== "") {
                $tickets->where("tickets.req_submitted", $filter["requirements"]);
            }
            if (isset($filter["type"]) && $filter["type"]) {
                switch ($filter["type"]) {
                    case '1':
                        $tickets->where("tickets.date_attend  IS NOT NULL");
                        $title = 'List of Attendees';
                        break;

                    case '2':
                        $tickets->where("tickets.date_leave IS NOT NULL");
                        $title = 'List of Early Out Attendees';
                        break;

                    case '3':
                        $tickets->where("tickets.date_attend IS NULL");
                        $title = 'List of Absentees';
                        break;

                    case '4':
                        $tickets->where("tickets.req_submitted", true);
                        $title = 'List with Submitted Requirements';
                        break;

                    case '5':
                        $tickets->where("tickets.req_submitted", false);
                        $title = 'List without Submitted Requirements';
                        break;
                }
            }
        }

        $branch = model('BranchModel')->find($filter["branch"] ?? null);

        $template = view('report/ajax/template', [
            'branchColumn'  => empty($filter["branch"]),
            'branch'        => $branch->branch_name ?? null,
            'tickets'       => $tickets->findAll(),
            'title'         => $title,
        ]);

        return $template;
    }

    public function indexx()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $mpdf = new Mpdf();

        // Define the Header/Footer before writing anything so they appear on the first page
        $mpdf->SetHTMLHeader('
        <div style="text-align: right; font-weight: bold;">
            My document
        </div>');
        $mpdf->SetHTMLFooter('
        <table width="100%">
            <tr>
                <td width="33%">{DATE j-m-Y}</td>
                <td width="33%" align="center">{PAGENO}/{nbpg}</td>
                <td width="33%" style="text-align: right;">My document</td>
            </tr>
        </table>');

        $mpdf->WriteHTML('Hello World');

        $mpdf->Output('filename.pdf', 'D');
    }

    public function ajaxList()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $tickets = $this->getTicketProvider();

        $tickets
            ->select(
                "tickets.id,
                events.event_code,
                tickets.ticket_number,
                branches.branch_code,
                branches.branch_name,
                tickets.id_number,
                tickets.first_name,
                tickets.last_name,
                tickets.middle_name,
                tickets.email,
                tickets.phone_number,
                tickets.position,
                tickets.department,
                tickets.year,
                tickets.track,
                tickets.req_submitted,
                tickets.date_attend,
                tickets.date_leave,
                tickets.created_at,
                encoder.username AS encoder_username,
                encoder.first_name AS encoder_first_name,
                encoder.last_name AS encoder_last_name,
                encoder.middle_name AS encoder_middle_name"
            )
            ->join('users AS encoder', 'tickets.created_by = encoder.id', 'left')
            ->join('events', 'tickets.event_id = events.id', 'inner')
            ->join('branches', 'tickets.branch_id = branches.id', 'left')
            ->where('events.active', true)
            ->where("tickets.deleted_at", null);

        try {
            if ($this->user->can('tickets.view-branch')) {
                $tickets->where('branches.id', $this->user->branch_id);
            }
        } catch (\ErrorException $ex) {
            $tickets->where('tickets.id', null);
        }

        $datatable = DataTable::of($tickets)
            ->filter(function ($tickets, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    $tickets
                        ->groupStart()
                        ->Like('tickets.id_number', $search)
                        ->orLike('tickets.first_name', $search)
                        ->orLike('tickets.last_name', $search)
                        ->orLike('tickets.email', $search)
                        ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (!empty($filter["branch"])) {
                        $tickets->where("branches.id", $filter["branch"]);
                    }
                    if (!empty($filter["position"])) {
                        $tickets->where("tickets.position", $filter["position"]);
                    }
                    if (!empty($filter["department"])) {
                        $tickets->where("tickets.department", $filter["department"]);
                    }
                    if (isset($filter["requirements"]) && $filter["requirements"] !== "") {
                        $tickets->where("tickets.req_submitted", $filter["requirements"]);
                    }
                    if (isset($filter["type"]) && $filter["type"] !== "") {
                        switch ($filter["type"]) {
                            case '1':
                                $tickets->where("tickets.date_attend <>", null);
                                break;

                            case '2':
                                $tickets->where("tickets.date_leave <>", null);
                                break;

                            case '3':
                                $tickets->where("tickets.date_attend", null);
                                break;
                        }
                    }
                }
            })
            ->add('name', function ($row) {
                return view("partials/elements/common/table_user_name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "email"       => $row->email,
                    "username"    => null,
                ]);
            })
            ->add('role', function ($row) {
                return view("partials/elements/tickets/role", [
                    "department"  => $row->department,
                    "position"    => $row->position,
                    "year"        => $row->year,
                    "track"       => $row->track,
                    "offcanvas"   => false
                ]);
            })
            ->add('requirements', function ($row) {
                return view("partials/elements/tickets/requirement_status", [
                    "status" => $row->req_submitted,
                ]);
            })
            ->format("date_attend", function ($value, $meta) {
                return $value ?? "Did not attend";
            })
            ->format("date_leave", function ($value, $meta) {
                return $value ?? "Did not leave early";
            })
            ->toJson(true);

        return $datatable;
    }

    /**
     * Gets the ticket provider
     */
    protected function getTicketProvider(): TicketModel
    {
        return new TicketModel();
    }

    /**
     * Gets the ticket provider
     */
    protected function getBranchProvider(): BranchModel
    {
        return new BranchModel();
    }
}
