<?php

namespace App\Models;

use CodeIgniter\Model;
use Config\Database;

class TicketModel extends Model
{
    protected $table            = 'tickets';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'object';
    protected $useSoftDeletes   = (ENVIRONMENT === 'production');
    protected $protectFields    = true;
    protected $allowedFields    = [
        'event_id',
        'ticket_number',
        'branch_id',
        'id_number',
        'first_name',
        'last_name',
        'middle_name',
        'email',
        'phone_number',
        'position',
        'department',
        'year',
        'track',
        'req_submitted',
        'date_attend',
        'date_leave',
        'created_by'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules      = [
        'id'                => [
            'label'         => 'ID',
            'rules'         => 'permit_empty',
        ],
        'event_id'          => [
            'label'         => 'Event',
            'rules'         => 'required|is_not_unique[events.id]',
        ],
        'ticket_number'     => [
            'label'         => 'Ticket number',
            'rules'         => 'required|is_unique[tickets.ticket_number,tickets.id,{id}]',
        ],
        'branch_id'         => [
            'label'         => 'Branch',
            'rules'         => 'required|is_not_unique[branches.id]',
        ],
        'id_number'         => [
            'label'         => 'Valid ID number',
            'rules'         => 'required|is_unique_combination[tickets.id_number,event_id,{event_id},branch_id,{branch_id},id,{id}]|max_length[30]',
        ],
        'first_name'        => [
            'label'         => 'First name',
            'rules'         => 'required|max_length[50]',
        ],
        'last_name'         => [
            'label'         => 'Last name',
            'rules'         => 'required|max_length[50]',
        ],
        'middle_name'       => [
            'label'         => 'Middle name',
            'rules'         => 'permit_empty|max_length[50]',
        ],
        'phone_number'      => [
            'label'         => 'Contact number',
            'rules'         => 'permit_empty|max_length[20]',
        ],
        'email'             => [
            'label'         => 'Email',
            'rules'         => 'required|max_length[254]|valid_email|is_unique_combination[tickets.email,event_id,{event_id},id,{id}]',
        ],
        'phone_number'      => [
            'label'         => 'Contact number',
            'rules'         => 'permit_empty|max_length[20]',
        ],
        'position'          => [
            'label'         => 'Position',
            'rules'         => 'required|max_length[20]',
        ],
        'department'        => [
            'label'         => 'Department',
            'rules'         => 'permit_empty|max_length[20]',
        ],
        'year'              => [
            'label'         => 'Year',
            'rules'         => 'permit_empty|max_length[20]',
        ],
        'track'             => [
            'label'         => 'Course/Strand',
            'rules'         => 'permit_empty|max_length[100]',
        ],
    ];
    protected $validationMessages   = [
        'event_id'          => [
            'required'      => '{field} is required.',
            'is_not_unique' => '{field} does not exist or might be inactive.',
        ],
        'ticket_number'     => [
            'required'      => '{field} is required.',
            'is_unique'     => 'You are attempting to register an already registered individual.',
        ],
        'branch_id'         => [
            'required'      => '{field} is required.',
            'is_not_unique' => '{field} does not exist.',
        ],
        'id_number'         => [
            'required'      => '{field} is required.',
            'is_unique_combination' => '{field} {value} already exist.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
        ],
        'first_name'        => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
        ],
        'last_name'         => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
        ],
        'middle_name'       => [
            'max_length'    => '{field} must only be {param} characters in maximum.',
        ],
        'email'             => [
            'required'      => '{field} is required.',
            'is_unique_combination' => '{field} {value} already exist.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
            'valid_email'   => '{value} is not a valid email.',
        ],
        'phone_number'      => [
            'max_length'    => '{field} must only be {param} characters in maximum.',
        ],
        'position'          => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
            'in_list'       => '{field} is not valid.',
        ],
        'department'        => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
            'in_list'       => '{field} is not valid.',
        ],
        'year'              => [
            'required'      => 'This is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
            'in_list'       => '{field} is not valid.',
        ],
        'track'             => [
            'required'      => 'This is required.',
            'max_length'    => '{field} must only be {param} characters in maximum.',
            'in_list'       => '{field} is not valid.',
        ],
    ];
    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = [];
    protected $afterInsert    = ['logInsert'];
    protected $beforeUpdate   = [];
    protected $afterUpdate    = ['logUpdate'];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = [];
    protected $afterDelete    = ['logDelete'];

    // Flags
    protected $skipLogAction  = false;

    /**
     * Deletes a single or multiple ticket.
     */
    public function deleteMultiple($id, $purge = false)
    {
        if (is_array($id)) {
            foreach ($id as $ticketId) {
                $this->delete($ticketId, $purge);
            }

            return true;
        }

        return $this->delete($id, $purge);
    }

    /**
     * Finds ticket with the specified ticket number
     */
    public function findByTicketNumber($ticketNumber)
    {
        return $this
            ->select(
                "tickets.id,
                tickets.event_id,
                events.event_code,
                events.event_name,
                tickets.ticket_number,
                branches.branch_code,
                branches.branch_name,
                tickets.id_number,
                tickets.first_name,
                tickets.last_name,
                tickets.middle_name,
                tickets.email,
                tickets.phone_number,
                tickets.position,
                tickets.department,
                tickets.year,
                tickets.track,
                tickets.req_submitted,
                tickets.date_attend,
                tickets.date_leave,
                tickets.created_at,
                encoder.username AS encoder_username,
                encoder.first_name AS encoder_first_name,
                encoder.last_name AS encoder_last_name,
                encoder.middle_name AS encoder_middle_name"
            )
            ->join('users AS encoder', 'tickets.created_by = encoder.id', 'left')
            ->join('events', 'tickets.event_id = events.id', 'inner')
            ->join('branches', 'tickets.branch_id = branches.id', 'left')
            ->where("tickets.deleted_at", null)
            ->where('ticket_number', $ticketNumber)
            ->first();
    }

    public function clockin($id)
    {
        // Database::connect()
        //     ->table('tickets')
        //     ->set('date_attend', 'CURRENT_TIMESTAMP', false)
        //     ->where('id', $id)
        //     ->where('date_attend', null)
        //     ->where('deleted_at', null)
        //     ->update();

        $ticket = $this->find($id);
        if ($ticket && empty($ticket->date_attend)) {
            $date = $this->setDate();
            if ($this->update($id, ['date_attend' => $date])) {
                $ticket->date_attend = $date;
            }
        }

        return $ticket;
    }

    public function clockout($id)
    {
        // Database::connect()
        //     ->table('tickets')
        //     ->set('date_leave', 'CURRENT_TIMESTAMP', false)
        //     ->where('id', $id)
        //     ->where('date_leave', null)
        //     ->where('deleted_at', null)
        //     ->update();

        $ticket = $this->find($id);
        if ($ticket && empty($ticket->date_leave)) {
            $date = $this->setDate();
            if ($this->update($id, ['date_leave' => $date,])) {
                $ticket->date_leave = $date;
            }
        }

        return $ticket;
    }

    // Log callback methods
    protected function logInsert(array $data)
    {
        $this->logAction('insert', $data['id']);
        return $data;
    }

    protected function logUpdate(array $data)
    {
        $action = 'update';

        if (!empty($data['data']['date_attend'])) {
            $action = 'time-in';
        } elseif (!empty($data['data']['date_leave'])) {
            $action = 'time-out';
        }

        $this->logAction($action, $data['id']);
        return $data;
    }

    protected function logDelete(array $data)
    {
        $this->logAction('delete', $data['id']);
        return $data;
    }

    protected function skipLog(callable $callback)
    {
        // Update skipLogAction flag first
        $this->skipLogAction = true;

        try {
            return $callback();
        } finally {
            // Reset to original state
            $this->skipLogAction = false;
        }
    }

    protected function logAction($action, $entityId)
    {
        if ($this->skipLogAction) {
            return;
        }

        $logsModel = new LogModel();
        $logsModel->logAction($action, $entityId, $this->table);
    }
}
