<?php

namespace Config;

use CodeIgniter\Config\BaseConfig;
use CodeIgniter\Validation\StrictRules\CreditCardRules;
use CodeIgniter\Validation\StrictRules\FileRules;
use CodeIgniter\Validation\StrictRules\FormatRules;
use CodeIgniter\Validation\StrictRules\Rules;

class Validation extends BaseConfig
{
    // --------------------------------------------------------------------
    // Setup
    // --------------------------------------------------------------------

    /**
     * Stores the classes that contain the
     * rules that are available.
     *
     * @var string[]
     */
    public array $ruleSets = [
        Rules::class,
        FormatRules::class,
        FileRules::class,
        CreditCardRules::class,
        \App\Validation\CustomRules::class,
    ];

    /**
     * Specifies the views that are used to display the
     * errors.
     *
     * @var array<string, string>
     */
    public array $templates = [
        'list'   => 'CodeIgniter\Validation\Views\list',
        'single' => 'CodeIgniter\Validation\Views\single',
    ];

    // --------------------------------------------------------------------
    // Rules
    // --------------------------------------------------------------------

    public array $registration = [
        'username'      => [
            'label'         => 'Auth.username',
            'rules'         => [
                'required',
                'max_length[30]',
                'min_length[3]',
                'alpha_numeric',
                'is_unique[users.username]',
            ],
            'errors'    => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
                'min_length'    => '{field} must be {param} characters in minimum.',
                'alpha_numeric' => '{field} must not contain special characters and whitespaces.',
                'is_unique'     => '{field} {value} already exists.',
            ],
        ],
        'branch_id'     => [
            'label'         => 'Branch',
            'rules'         => [
                'permit_empty',
                'is_not_unique[branches.id]'
            ],
            'errors'        => [
                'is_not_unique' => '{field} does not exist.',
            ]
        ],
        'first_name'    => [
            'label'         => 'First name',
            'rules'         => [
                'required',
                'max_length[50]',
            ],
            'errors'        => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
            ],
        ],
        'last_name'     => [
            'label'         => 'Last name',
            'rules'         => [
                'required',
                'max_length[50]',
            ],
            'errors'        => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
            ],
        ],
        'middle_name'   => [
            'label'         => 'Middle name',
            'rules'         => [
                'permit_empty',
                'max_length[50]',
            ],
            'errors'        => [
                'max_length'    => '{field} must only contain {param} characters in maximum.',
            ],
        ],
        'phone_number'  => [
            'label'         => 'Contact number',
            'rules'         => [
                'permit_empty',
                'max_length[20]',
            ],
            'errors'        => [
                'max_length'    => '{field} must only contain {param} characters in maximum.',
            ],
        ],
        'email' => [
            'label' => 'Auth.email',
            'rules' => [
                'required',
                'max_length[254]',
                'valid_email',
                'is_unique[auth_identities.secret]',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
                'valid_email'   => '{value} is not a valid email.',
                'is_unique'     => '{field} {value} already exists.',
            ],
        ],
        'password' => [
            'label' => 'Auth.password',
            'rules' => [
                'required',
                'max_byte[72]',
                'strong_password[]',
            ],
            'errors' => [
                'required'      => '{field} is required.',
                'max_byte'      => 'Auth.errorPasswordTooLongBytes',
            ]
        ],
        'password_confirm' => [
            'label' => 'Auth.passwordConfirm',
            'rules' => [
                'required',
                'matches[password]',
            ],
            'errors'   => [
                'required'      => 'Please confirm password.',
                'matches'       => 'Password did not match.',
            ],
        ],
    ];

    public array $userIdValidation = [
        'id' => [
            'label' => 'ID',
            'rules' => [
                'required',
                'numeric',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'numeric'       => '{field} must be numeric.',
            ],
        ],
    ];

    public array $usernameValidation = [
        'username' => [
            'label' => 'Auth.username',
            'rules' => [
                'required',
                'max_length[30]',
                'min_length[3]',
                'alpha_numeric',
                'is_unique[users.username,id,{id}]',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
                'min_length'    => '{field} must be {param} characters in minimum.',
                'alpha_numeric' => '{field} must not contain special characters and whitespaces.',
                'is_unique'     => '{field} {value} already exists.',
            ],
        ],
    ];

    public array $emailValidation = [
        'email' => [
            'label' => 'Auth.email',
            'rules' => [
                'required',
                'max_length[254]',
                'valid_email',
                'is_unique[auth_identities.secret,user_id,{id}]',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
                'valid_email'   => '{field} is not a valid email.',
                'is_unique'     => '{field} {value} already exists.',
            ],
        ],
    ];

    public array $groupValidation = [
        'group' => [
            'label' => 'Group',
            'rules' => [
                'required',
                'max_length[255]',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'max_length'    => '{field} must only contain {param} characters in maximum.',
            ],
        ],
    ];

    public array $passwordValidation = [
        'password' => [
            'label' => 'Auth.password',
            'rules' => [
                'permit_empty',
                'max_byte[72]',
                'strong_password[]',
            ],
            'errors' => [
                'max_byte'      => 'Auth.errorPasswordTooLongBytes',
            ]
        ],
        'password_confirm' => [
            'label' => 'Auth.passwordConfirm',
            'rules' => [
                'matches[password]',
            ],
            'errors'   => [
                'matches'       => 'Password did not match or is empty.',
            ],
        ],
    ];

    public array $login = [
        'login'         => [
            'label'         => 'Email or Username',
            'rules'         => [
                'required',
            ],
            'errors'    => [
                'required'      => '{field} is required.',
            ],
        ],
        'username'      => [
            'label'         => 'Auth.username',
            'rules'         => [
                'required',
                'regex_match[/\A[a-zA-Z0-9\.]+\z/]',
            ],
            'errors'    => [
                'required'      => '{field} is required.',
                'regex_match'   => '{field} is invalid.'
            ],
        ],
        'email' => [
            'label' => 'Auth.email',
            'rules' => [
                'required',
                'valid_email',
            ],
            'errors'   => [
                'required'      => '{field} is required.',
                'valid_email'   => '{field} is not a valid email.',
            ],
        ],
        'password' => [
            'label' => 'Auth.password',
            'rules' => [
                'required',
                'max_byte[72]',
                'strong_password[]',
            ],
            'errors' => [
                'required'      => '{field} is required.',
                'max_byte'      => 'Auth.errorPasswordTooLongBytes',
            ]
        ],
    ];
}
