<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use App\Models\LogModel;
use Hermawan\DataTables\DataTable;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Shield\Entities\User;

class LogController extends BaseController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "log";

    protected array $data = [
        "searchbar" => false,
        "statusbar" => false,
        "module"    => "System Logs",
    ];

    // Entities
    protected ?User $user = null;

    public function __construct()
    {
        helper(["form", "text", "log"]);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function index()
    {
        if (!$this->user->can('super.access', 'logs.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        return view("{$this->dir}/post/list", $this->data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'logs.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxList();
        }

        //$data = $this->postList();

        return view("{$this->dir}/post/list", $this->data);
    }

    protected function ajaxList()
    {
        $logs = $this->getLogProvider();

        $logs
            ->select(
                "logs.id,
                logs.timestamp,
                logs.action,
                logs.entity_id,
                logs.module,
                users.username,
                users.first_name,
                users.last_name,
                users.middle_name,"
            )
            ->join("users", "logs.performed_by = users.id", "left")
            ->where("users.deleted_at", null);

        $datatable = DataTable::of($logs)
            ->filter(function ($logs, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    // $logs
                    //     ->groupStart()
                    //     ->like('events.event_code', $search)
                    //     ->orLike('events.event_name', $search)
                    //     ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (!empty($filter["module"])) {
                        $logs->where("logs.module", $filter["module"]);
                    }
                }
            })
            ->format('timestamp', function ($value, $meta) {
                return date("Y-M-d | h:i:s A", strtotime($value));
            })
            ->format('module', function ($value, $meta) {
                return ucfirst($value);
            })
            ->add('description', function ($row) {
                return log_descriptor($row);
            })
            ->add('performed_by', function ($row) {
                return view("partials/elements/common/table_user_name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "username"    => $row->username,
                ]);
            })
            ->toJson(true);

        return $datatable;
    }

    public function clear()
    {
        $logs = $this->getLogProvider();
        $logs->truncateTable();

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Logs Cleared!",
        ]);
    }

    protected function getLogProvider(): LogModel
    {
        return new LogModel();
    }
}
