<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;
use App\Models\TicketModel;
use App\Models\EventModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\Encryption\Exceptions\EncryptionException;
use CodeIgniter\Shield\Entities\User;
use Config\RiceData;
use Config\Services;
use Hermawan\DataTables\DataTable;
use App\Libraries\Gmailer;

class TicketController extends BaseController
{
    protected bool $isAjaxProcess = true;

    protected string $dir = "ticket";

    protected array $data = [
        "searchbar" => true,
        "statusbar" => true,
        "module"    => "Ticket Management",
    ];

    // Models
    protected ?TicketModel $tickets = null;
    protected ?EventModel $events   = null;

    // Entities
    protected ?User $user           = null;

    // Config
    protected ?RiceData $rices      = null;

    /**
     * Format ticket number based on event code, branch code
     * and id number
     * 
     * Example: AU_Evnt.2k24-JSC-2101488
     */
    protected const EVNT_BRNCH_ID_FORMAT = 1;

    /**
     * Format ticket number based on microtime in decimal
     * 
     * Example: 1801836324002721
     */
    protected const DEC_MICRO_TIME_FORMAT = 2;

    /**
     * Format ticket number based on date and time
     * 
     * Example: 20240617024342
     */
    protected const DEC_TIME_DATE_FORMAT = 3;

    /**
     * Format ticket number based on branch code, id number, and microtime
     * 
     * Example: JSC-A1B2C3D4E5
     */
    protected const HASH_MCRTIME_ID_FORMAT = 4;

    /**
     * Format ticket number based on branch code, id number, and microtime
     * 
     * Example: JSC-67B4609E2F3FA
     */
    protected const UNIQID_UPPER_FORMAT = 5;

    public function __construct()
    {
        helper(["form", "input", "qrcode", "carray"]);

        if (!$this->user = auth()->user()) {
            throw PageNotFoundException::forPageNotFound();
        }
    }

    public function index()
    {
        if (!$this->user->can('super.access', 'tickets.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/list", $this->data);
        }

        //$data = array_merge($this->data, $this->postList());

        return view("{$this->dir}/post/list", $this->data);
    }

    public function listAction()
    {
        if (!$this->user->can('super.access', 'tickets.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxList();
        }

        //$data = $this->postList();
        $data = [];

        return view("{$this->dir}/post/list", $data);
    }

    protected function ajaxList()
    {
        $tickets = $this->getTicketProvider();

        $tickets
            ->select(
                "tickets.id,
                events.event_code,
                tickets.ticket_number,
                branches.branch_code,
                branches.branch_name,
                tickets.id_number,
                tickets.first_name,
                tickets.last_name,
                tickets.middle_name,
                tickets.email,
                tickets.phone_number,
                tickets.position,
                tickets.department,
                tickets.year,
                tickets.track,
                tickets.req_submitted,
                tickets.date_attend,
                tickets.date_leave,
                tickets.created_at,
                tickets.updated_at,
                tickets.deleted_at,
                encoder.username AS encoder_username,
                encoder.first_name AS encoder_first_name,
                encoder.last_name AS encoder_last_name,
                encoder.middle_name AS encoder_middle_name"
            )
            ->join('users AS encoder', 'tickets.created_by = encoder.id', 'left')
            ->join('events', 'tickets.event_id = events.id', 'inner')
            ->join('branches', 'tickets.branch_id = branches.id', 'left')
            ->where('events.active', true);

        if (!$this->user) {
            $tickets->where('tickets.id', null);
        } elseif (!$this->user->can('tickets.view-all')) {

            $tickets->where("tickets.deleted_at", null);

            if ($this->user->can('tickets.view-branch')) {
                $tickets->where('branches.id', $this->user->branch_id);
            } else {
                $tickets->where('tickets.created_by', $this->user->id);
            }
        }

        $datatable = DataTable::of($tickets)
            ->filter(function ($tickets, $request) {

                if (isset($request->new_search)) {
                    $search = $request->new_search;
                    $tickets
                        ->groupStart()
                        ->Like('tickets.id_number', $search)
                        ->orLike('tickets.ticket_number', $search)
                        ->orLike('tickets.first_name', $search)
                        ->orLike('tickets.last_name', $search)
                        ->orLike('tickets.email', $search)
                        ->groupEnd();
                }

                if (isset($request->filter)) {

                    parse_str($request->filter, $filter);

                    if (!empty($filter["branch"])) {
                        $tickets->where("branches.id", $filter["branch"]);
                    }
                    if (!empty($filter["position"])) {
                        $tickets->where("tickets.position", $filter["position"]);
                    }
                    if (!empty($filter["department"])) {
                        $tickets->where("tickets.department", $filter["department"]);
                    }
                    if (isset($filter["requirements"]) && $filter["requirements"] !== "") {
                        $tickets->where("tickets.req_submitted", $filter["requirements"]);
                    }
                }
            })
            // ->edit('id', function ($row) {
            //     return view("partials/elements/common/id", [
            //         "id"          => $row->id,
            //         "deleted"     => $row->deleted_at,
            //     ]);
            // })
            ->edit('ticket_number', function ($row) {
                return '<div class="marquee" style="width:150px;height:1.25rem;"><span>' . esc($row->ticket_number) . '</span></div>';
            })
            ->format('updated_at', function ($value) {
                return date('M j, Y h:i A', strtotime($value));
            })
            ->add('name', function ($row) {
                return view("partials/elements/tickets/name", [
                    "first_name"  => $row->first_name,
                    "middle_name" => $row->middle_name,
                    "last_name"   => $row->last_name,
                    "email"       => $row->email,
                    "id_number"   => $row->id_number,
                    "username"    => null,
                    // "deleted"     => $row->deleted_at,
                    "deleted"     => null,
                ]);
            })
            ->add('role', function ($row) {
                return view("partials/elements/tickets/role", [
                    "department"  => $row->department,
                    "position"    => $row->position,
                    "year"        => $row->year,
                    "track"       => $row->track,
                    "offcanvas"   => false
                ]);
            })
            ->add('requirements', function ($row) {
                return view("partials/elements/tickets/requirement_status", [
                    "status" => $row->req_submitted,
                ]);
            })
            ->add('attended', function ($row) {
                return view("partials/elements/common/active_status", [
                    "active"    => !empty($row->date_attend)
                ]);
            })
            ->add('leaved', function ($row) {
                return view("partials/elements/common/active_status", [
                    "active"    => !empty($row->date_leave)
                ]);
            })
            ->add('action', function ($row) {
                return view("partials/elements/tickets/action_buttons", [
                    "ticket"      => $row,
                    "offcanvas"   => true,
                    "permissions" => [
                        "edit"    => "tickets.edit",
                        "delete"  => "tickets.delete",
                    ],
                ]);
            })
            ->addNumbering();

        return $datatable->toJson(true);
    }

    public function createView()
    {
        if (!$this->user->can('super.access', 'tickets.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->getActiveEventId() === null) {
            return redirect()->to(ticket_url())->with("warning", "There is no active events found.");
        }

        $this->data["searchbar"] = false;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/create", $this->data);
        }

        return view("{$this->dir}/post/create", $this->data);
    }

    public function createAction()
    {
        if (!$this->user->can('super.access', 'tickets.create')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxCreate();
        }

        //return $this->postCreate();
    }

    protected function ajaxCreate()
    {
        $tickets    = $this->getTicketProvider();
        $capitalize = ['first_name', 'middle_name', 'last_name'];
        $data       = clean_input_data($this->request->getPost(), $capitalize);

        try {
            $data['created_by'] = $this->user->id;
        } catch (\ErrorException $ex) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "You are not logged in. Please logged in first.",
            ]);
        }

        $eventId = $this->getActiveEventId();

        if (!$eventId) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "There is no currently active event found.",
            ]);
        }

        $data['event_id']       = $eventId;
        $data['ticket_number']  = $this->setTicketNumber($data, self::HASH_MCRTIME_ID_FORMAT);
        $data['req_submitted']  = isset($data['req_submitted']);

        $rules  = $this->getValidations($data);
        $msgs   = $tickets->getValidationMessages();

        if (!$this->validateData($data, $rules, $msgs)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $this->validator->getErrors(),
                "data"      => $this->validator->getError('ticket_number'),
            ]);
        }

        $tickets->save($data);

        try {
            if ($this->sendQRCode($tickets->getInsertID())) {
                session()->setFlashdata("success", "Ticket has been successfully registered.");
            } else {
                session()->setFlashdata("warning", "The ticket QR code did not send. Please try resending.");
            }
        } catch (\ErrorException $ex) {
            log_message('error', $ex);
            session()->setFlashdata("error", "Something went wrong in sending QR code. Please try resending.");
        }

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Ticket has been successfully registered.",
        ]);
    }

    protected function postCreate()
    {
        $tickets = $this->getTicketProvider();

        $data = $this->request->getPost();
        $data['created_by'] = auth()->user()->id;

        if (!$tickets->save($data)) {
            return redirect()->back()->withInput()->with('error', $tickets->errors());
        }

        return redirect()->to('ticket')->with('success', 'Ticket has been successfully added!');
    }

    public function editView($id)
    {
        if (!$this->user->can('super.access', 'tickets.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $tickets    = $this->getTicketProvider();
        $ticket     = $tickets->find($id);

        if ($ticket === null) {
            return redirect()->back()->with("error", "There is a problem fetching the ticket data. Ticket cannot be edit right now.");
        }

        if (!$this->isEventActive($ticket->event_id)) {
            return redirect()->back()->with("warning", "The ticket's event is not active. Ticket cannot be edit right now.");
        }

        $this->data["searchbar"] = false;
        $this->data["ticket"]    = $ticket;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/edit", $this->data);
        }

        return view("{$this->dir}/post/edit", $this->data);
    }

    public function editAction($id = null)
    {
        if (!$this->user->can('super.access', 'tickets.edit')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxEdit($id);
        }

        //return $this->postEdit();
    }

    protected function ajaxEdit($id = null)
    {
        $capitalize     = ['first_name', 'middle_name', 'last_name'];
        $data           = clean_input_data($this->request->getPost(), $capitalize);
        $data['id']     = $data['id'] ?? $id;
        $tickets        = $this->getTicketProvider();
        $ticket         = $tickets->find($data['id']);

        if ($ticket === null) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "There is a problem fetching ticket data. Ticket cannot be edit right now.",
            ]);
        }

        if (!$this->isEventActive($ticket->event_id)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => null,
                "data"      => "The ticket's event is not active. Ticket cannot be edit right now.",
            ]);
        }

        $data['req_submitted'] = isset($data['req_submitted']);

        $data['branch_id'] = $ticket->branch_id;

        $rules  = $this->getValidations($data, 'update');
        $msgs   = $tickets->getValidationMessages();

        if (!$this->validateData($data, $rules, $msgs)) {
            return $this->response->setJSON([
                "success"   => false,
                "errors"    => $this->validator->getErrors(),
                "data"      => $this->validator->getError('ticket_number'),
            ]);
        }

        $tickets->save($data);

        if ($ticket->email !== $data['email']) {
            try {
                if ($this->sendQRCode($ticket->id)) {
                    session()->setFlashdata("success", "Ticket has been successfully updated and QR code has been resent.");
                } else {
                    session()->setFlashdata("warning", "The ticket QR code did not send. Please try resending.");
                }
            } catch (\ErrorException $ex) {
                log_message('error', $ex);
                session()->setFlashdata("error", "Something went wrong in sending QR code. Please try resending.");
            }
        } else {
            session()->setFlashdata("success", "Ticket has been successfully updated.");
        }

        return $this->response->setJSON([
            "success"   => true,
            "errors"    => null,
            "data"      => "Ticket has been successfully updated.",
        ]);
    }

    public function validationView($ticket = null)
    {
        if (!$this->user->can('super.access', 'tickets.validate')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $this->data["searchbar"]    = false;
        $this->data["statusbar"]    = false;
        $this->data["module"]       = "Validate Ticket";
        $this->data["ticket_code"]  = $ticket;

        if ($this->isAjaxProcess) {
            return view("{$this->dir}/ajax/validation", $this->data);
        }

        return view("{$this->dir}/post/validation", $this->data);
    }

    public function validationAction()
    {
        if (!$this->user->can('super.access', 'tickets.validate')) {
            throw PageNotFoundException::forPageNotFound();
        }

        if ($this->isAjaxProcess) {
            return $this->ajaxValidation();
        }

        //return $this->postValidation();
    }

    protected function ajaxValidation()
    {
        $ticketCode     = $this->request->getPost('ticketCode');
        $ticketNumber   = $this->verifyTicket($ticketCode);
        $tickets        = $this->getTicketProvider();
        $ticket         = $tickets->findByTicketNumber($ticketNumber, $count);

        if (!$ticket) {
            return $this->response->setJSON([
                'success'   => false,
                'data'      => 'Ticket Number/QR Code cannot be found.'
            ]);
        }

        $output = [];

        foreach ($ticket as $item) {
            if (!$this->isEventActive($item->event_id)) {
                continue;
            }

            $name      = [];
            $name[]    = $item->first_name;
            $name[]    = !empty($item->middle_name) ? $item->middle_name[0] . '.' : null;
            $name[]    = $item->last_name;

            $fullname  = implode(' ', array_filter($name));

            $position  = empty($item->department) ? $item->position : "{$item->department} {$item->position}";
            $position .= empty($item->year) ? '' : " | {$item->year}";
            $position .= empty($item->track) ? '' : (empty($item->year) ? " | {$item->track}" : "-{$item->track}");

            $output = [
                'ticketID'      => $item->id,
                'ticketNumber'  => $item->ticket_number,
                'eventName'     => $item->event_name,
                'branch'        => $item->branch_name,
                'idNumber'      => $item->id_number,
                'fullname'      => $fullname,
                'position'      => strtoupper($position),
                'requirements'  => $item->req_submitted,
                'timeAttended'  => $item->date_attend,
                'timeLeaved'    => $item->date_leave,
            ];

            break;
        }

        return $this->response->setJSON([
            'success'   => true,
            'data'      => $output,
            'count'     => $count
        ]);
    }

    public function trackTime()
    {
        if (!$this->user->can('super.access', 'tickets.validate')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = $this->request->getPost('ticketID');
        $action     = $this->request->getPost('action');
        $tickets    = $this->getTicketProvider();
        $result     = null;

        switch ($action) {

            case 'CLOCKIN':
                $result = $tickets->clockin($id);
                break;

            case 'CLOCKOUT':
                $result = $tickets->clockout($id);
                break;
        }

        if (!empty($result)) {

            $output = [
                'requirements'  => $result->req_submitted,
                'timeAttended'  => $result->date_attend,
                'timeLeaved'    => $result->date_leave,
            ];

            return $this->response->setJSON([
                'success'       => true,
                'data'          => $output
            ]);
        } else {

            return $this->response->setStatusCode(400)->setJSON([
                'success'       => false,
                'data'          => 'Operation failed.'
            ]);
        }
    }

    protected function verifyTicket($ticketCode)
    {
        try {
            $ticketNumber = decrypt_data($ticketCode);
            return $ticketNumber;
        } catch (EncryptionException $ex) {
            //throw $ex;
        }

        $endSegment = '';

        try {
            $segments = explode('/', parse_url($ticketCode, PHP_URL_PATH));
            $endSegment = end($segments);
            return decrypt_data($endSegment);
        } catch (EncryptionException $ex) {
            //throw $ex;
        }

        return $endSegment;
    }

    public function delete($id = null)
    {
        if (!$this->user->can('super.access', 'tickets.delete')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $id         = json_decode($this->request->getVar("id")) ?? $id ?? 0;
        $tickets    = $this->getTicketProvider();
        $purge      = $this->user->can('super.access', 'admin.settings');

        if ($tickets->deleteMultiple($id, $purge)) {
            return $this->response->setJSON([
                "success"   => true,
                "title"     => "Ticket(s) Deleted",
                "data"      => "Ticket(s) has been deleted."
            ]);
        }

        return $this->response->setJSON([
            "success"   => false,
            "title"     => "Deletion Failed",
            "data"      => "Ticket(s) cannot be deleted."
        ]);
    }

    public function resendQRCode($_id = null)
    {
        if (!$this->user->can('super.access', 'tickets.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $data = json_decode($this->request->getVar("id")) ?? $_id;
        $success = true;

        try {
            if ($data === null) {
                throw new \ErrorException("ID is null. Resending QR Code cannot be done.");
            }

            if (is_array($data)) {
                $errors = [];
                foreach ($data as $id) {
                    if (!$this->sendQRCode($id)) {
                        $errors[] = $id;
                    }
                }

                $output = "Ticket QR Code has been resent to their respective owner's email.";

                if (count($errors) > 0) {
                    $idWithErrors = implode(", ", $errors);
                    $output = "Some ticket QR Code has not been send. [{$idWithErrors}]";
                    $success = false;
                }
            } else {
                $id = $data;

                $output = "Ticket QR Code has been resent to owner's email.";

                if (!$this->sendQRCode($id)) {
                    throw new \ErrorException("QR Code did not send.");
                }
            }

            return $this->response->setJSON([
                "success"   => $success,
                "title"     => "QR Code Resent",
                "data"      => $output
            ]);
        } catch (\ErrorException $ex) {
            log_message('error', $ex);
        }

        return $this->response->setJSON([
            "success"   => false,
            "title"     => "Resending Failed",
            "data"      => "Ticket QR Code cannot be resend."
        ]);
    }

    protected function sendQRCode($ticket_id): bool
    {
        $tickets        = $this->getTicketProvider();
        $events         = $this->getEventProvider();

        $ticket         = $tickets->find($ticket_id);

        if ($ticket === null) {
            throw new \ErrorException("Ticket cannot be found.");
        }

        $template = file_get_contents(APPPATH . '/Views/email/qr_code.php');
        if ($template === false) {
            return 'Error loading email template.';
        }

        $event = $events->find($ticket->event_id);
        $mailer = Services::email();
        $to_email = $ticket->email ? explode('|', $ticket->email)[0] : '';
        $banners = [];

        $images = [
            'dos-and-donts.jpg',
            'enrollment-1.jpg',
            'enrollment-2.jpg',
            'enrollment-3.jpg',
        ];

        foreach ($images as $image) {
            $banners[] = view('email/banner', ['img' => base_url(['assets', 'img', 'banners', $image])]);
        }

        $data = [
            '{{aulogo}}'    => base_url('assets/img/au_logo.png'),
            '{{name}}'      => $ticket->first_name,
            '{{event}}'     => $event->event_name,
            '{{qrcode}}'    => generate_qr_code_data($ticket->ticket_number, 150, true),
            '{{banners}}'   => implode(' ', $banners)
        ];
        $body = strtr($template, $data);

        $mailer
            ->setTo($to_email)
            ->setSubject('Your Ticket Has Been Registered')
            ->setMessage($body)
            ->setMailType('html');

        return $mailer->send();
    }

    // protected function sendQRCode($ticket_id, $bcc_email = null): bool
    // {
    //     $tickets        = $this->getTicketProvider();
    //     $events         = $this->getEventProvider();

    //     $ticket         = $tickets->find($ticket_id);

    //     if ($ticket === null) {
    //         throw new \ErrorException("Ticket cannot be found.");
    //     }

    //     $event          = $events->find($ticket->event_id);
    //     $mailer         = new \App\Libraries\SendGridService();
    //     $statuscode     = 0;
    //     $to_email       = $ticket->email ? explode('|', $ticket->email)[0] : '';
    //     $to_name        = $ticket->first_name;
    //     $current_event  = $event->event_name;
    //     $qr_code_url    = generate_qr_code_data($ticket->ticket_number, 150, true);

    //     try {
    //         $statuscode = $mailer->sendQRCode($to_email, $to_name, $current_event, $qr_code_url);
    //     } catch (\Throwable $th) {
    //         $tickets->update($ticket->id, ['email' => $to_email . '|qr fail']);
    //         throw $th;
    //     }

    //     if ($statuscode == 202) {
    //         $tickets->update($ticket->id, ['email' => $to_email]);
    //         return true;
    //     } else {
    //         $tickets->update($ticket->id, ['email' => $to_email . '|qr fail']);
    //         return false;
    //     }
    // }

    protected function setTicketNumber(array $data, $format = self::EVNT_BRNCH_ID_FORMAT)
    {
        switch ($format) {

            case self::DEC_MICRO_TIME_FORMAT:
                return hexdec(uniqid());

            case self::DEC_TIME_DATE_FORMAT:
                return date("YmdHis");

            case self::HASH_MCRTIME_ID_FORMAT:
                $branch         = model("BranchModel")->find($data["branch_id"]);
                $branchCode     = $branch ? $branch->branch_code : "";

                $idNumber       = trim($data["id_number"]);
                $uniqueString   = $idNumber . microtime();
                $hash           = md5($uniqueString);
                $trim           = substr($hash, 0, 10);

                return $branchCode . '-' . strtoupper($trim);

            case self::UNIQID_UPPER_FORMAT:
                $branch         = model("BranchModel")->find($data["branch_id"]);
                $branchCode     = $branch ? $branch->branch_code : "";

                return strtoupper(uniqid($branchCode ? $branchCode . '-' : ''));

            default:
                $event      = trim($data["event_id"] ?? '') ? model("EventModel")->find($data["event_id"]) : null;
                $eventCode  = $event ? $event->event_code : "";

                $branch     = trim($data["branch_id"] ?? '') ? model("BranchModel")->find($data["branch_id"]) : null;
                $branchCode = $branch ? $branch->branch_code : "";

                $idNumber   = trim($data["id_number"] ?? "");

                $segment = [];

                $segment[] = str_replace(["-", " "], "", $eventCode  ?? "");
                $segment[] = str_replace(["-", " "], "", $branchCode ?? "");
                $segment[] = str_replace(["-", " "], "", $idNumber);

                // Filter null or empty elements out
                $filtered = array_filter($segment, fn($value) => !is_null($value) && $value !== '');

                return implode("-", $filtered);
        }
    }

    public function fetchOptions($type, $key, $secondary = null)
    {
        if (!$this->user->can('super.access', 'tickets.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $rice = $this->getRiceConfig();

        switch ($type) {
            case 'position':
                $options = $rice->getDepartmentList($key);
                break;
            case 'department':
                $options = $rice->getYearAndTrackList($key, $secondary);
                break;
            default:
                $options = [];
        }

        return $this->response->setJSON($options);
    }

    public function fetchDynamic()
    {
        if (!$this->user->can('super.access', 'tickets.manage')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $rice = $this->getRiceConfig();

        return $this->response->setJSON($rice->getAllSelection());
    }

    /**
     * Gets the rice data config
     */
    protected function getRiceConfig(): RiceData
    {
        if ($this->rices === null) {
            $this->rices = new RiceData();
        }

        return $this->rices;
    }

    /**
     * Gets the event provider
     */
    protected function getEventProvider(): EventModel
    {
        if ($this->events === null) {
            $this->events = new EventModel();
        }

        return $this->events;
    }

    /**
     * Gets the ticket provider
     */
    protected function getTicketProvider(): TicketModel
    {
        if ($this->tickets === null) {
            $this->tickets = new TicketModel();
        }

        return $this->tickets;
    }

    /**
     * Gets the id of an active event
     * 
     * @return int|string|null
     */
    protected function getActiveEventId()
    {
        $events = $this->getEventProvider();

        if ($event = $events->findActiveEvent()) {
            return $event->id;
        }

        return null;
    }

    /**
     * Checks if event is active
     */
    protected function isEventActive($id): bool
    {
        $events = $this->getEventProvider();

        return $events->isActive($id);
    }

    /**
     * Returns the model's defined validation rules with slight modification for insert.
     */
    protected function getValidations(array &$data, string $action = 'insert'): array
    {
        $tickets    = $this->getTicketProvider();
        $rices      = $this->getRiceConfig();
        $rules      = $tickets->getValidationRules();

        if ($action === 'update') {
            foreach (['event_id', 'ticket_number'] as $i) {
                unset($rules[$i]);
            }
        }

        $positions  = $rices->getPositionList();
        $posList    = implode(",", array_keys($positions));
        $rules['position']['rules'] = "required|max_length[20]|in_list[{$posList}]";

        if (empty($data['position'])) {
            return $rules;
        }

        $departments = $rices->getDepartmentList($data['position']);
        if (empty($departments)) {
            $this->setKeys($data, null, 'department', 'year', 'track');
            return $rules;
        }

        $deptList = implode(",", array_keys($departments));
        $rules['department']['rules'] = "required|max_length[20]|in_list[{$deptList}]";

        if (empty($data['department'])) {
            return $rules;
        }

        $yearTrack = $rices->getYearAndTrackList($data['position'], $data['department']);

        if (empty($yearTrack['years'])) {
            $this->setKeys($data, null, 'year');
        } else {
            $yearList   = implode(",", $yearTrack['years']);
            $rules['year']['rules'] = "required|max_length[20]|in_list[{$yearList}]";
        }

        if (empty($yearTrack['tracks'])) {
            $this->setKeys($data, null, 'track');
        } else {
            $trackList  = implode(",", array_keys($yearTrack['tracks']));
            $rules['track']['rules'] = "required|max_length[20]|in_list[{$trackList}]";
        }

        return $rules;
    }

    protected function setKeys(array &$data, $value, ...$keys)
    {
        foreach ($keys as $key) {
            $data[$key] = $value;
        }
    }



    public function ticketConflictChecker()
    {
        if (!$this->user->can('super.access', 'admin.access')) {
            throw PageNotFoundException::forPageNotFound();
        }

        $db = \Config\Database::connect();
        $builder = $db->table('tickets');

        // Subquery to find duplicate ticket numbers
        $subquery1 = $db->table('tickets')
            ->select('ticket_number')
            ->groupBy('ticket_number')
            ->having('COUNT(*) >', 1);

        // Subquery to get the minimum `id` of each duplicate ticket_number
        $subquery2 = $db->table('tickets')
            ->select('MIN(id)', false)
            ->groupBy('ticket_number')
            ->having('COUNT(*) >', 1);

        // Main query
        $query = $builder->whereIn('ticket_number', $subquery1, false)
            ->whereNotIn('id', $subquery2, false)
            ->get();

        // Fetch results
        return response()->setJSON($query->getResult());
    }

    public function ticketConflictFixer() {}
}
