<?php

namespace App\Controllers\Shield;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\HTTP\RedirectResponse;
use CodeIgniter\Shield\Controllers\LoginController as ShieldLogin;

class LoginController extends ShieldLogin
{
    /**
     * Attempts to log the user in.
     */
    public function LoginAction(): RedirectResponse
    {
        helper('array');

        $data           = $this->request->getPost();
        $field          = 'login';
        $login          = !empty($data['login']) ? $data['login'] : null;

        if ($login !== null) {
            $field = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
            $data[$field] = $login;
            unset($data['login']);
        }

        // Validate here first, since some things,
        // like the password, can only be validated properly here.
        $rules = $this->checkRules($field, $this->getValidationRules());

        if (!$this->validateData($data, $rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        /** @var array $credentials */
        $credentials             = array_filter($data, function ($key) {
            return in_array($key, setting('Auth.validFields'));
        }, ARRAY_FILTER_USE_KEY);

        $credentials['password'] = $this->request->getPost('password');
        $remember                = (bool) $this->request->getPost('remember');

        /** @var Session $authenticator */
        $authenticator = auth('session')->getAuthenticator();

        // Attempt to login
        $result = $authenticator->remember($remember)->attempt($credentials);
        if (!$result->isOK()) {
            $errorMessage = $this->getCustomErrorMessage($result->reason());
            return redirect()->route('login')->withInput()->with('error', $errorMessage);
        }

        // If an action has been defined for login, start it up.
        if ($authenticator->hasAction()) {
            return redirect()->route('auth-action-show')->withCookies();
        }

        return redirect()->to(config('Auth')->loginRedirect())->withCookies();
    }

    /** Check rules and unset some unnecessary ones */
    protected function checkRules(string $field, array $rules): array
    {
        switch ($field) {
            case 'email':
                unset($rules['login']);
                unset($rules['username']);
                break;

            case 'username':
                unset($rules['login']);
                unset($rules['email']);
                break;

            default:
                unset($rules['email']);
                unset($rules['username']);
                break;
        }

        return $rules;

        // log_message('debug', json_encode($rules));
    }

    /**
     * Get a custom error message based on the reason.
     */
    protected function getCustomErrorMessage(string $reason): string
    {
        switch ($reason) {
            case 'Unable to log you in. Please check your credentials.':
            case 'Unable to log you in. Please check your password.':
            case 'Can not log you in as you are currently banned.':
                return 'Incorrect credentials or account is disabled.';
            case 'You have been logged out because you have been banned.':
                return 'Sorry, you were logged out because your account was disabled.';
            default:
                return $reason;
        }
    }
}
