<?php

namespace App\Models;

use CodeIgniter\Model;

class BranchModel extends Model
{
    protected $table            = 'branches';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'object';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = ['branch_code', 'branch_name', 'active', 'created_by'];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules      = [
        'id'            => [
            'label'         => 'ID',
            'rules'         => 'permit_empty',
        ],
        'branch_code'   => [
            'label'         => 'Branch code',
            'rules'         => 'required|is_unique[branches.branch_code,id,{id}]|max_length[20]|alpha_dash',
        ],
        'branch_name'   => [
            'label'         => 'Branch name',
            'rules'         => 'required|max_length[30]|min_length[8]',
        ],
    ];
    protected $validationMessages   = [
        'branch_code'   => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only contain {param} characters in maximum.',
            'is_unique'     => '{field} {value} already exists.',
        ],
        'branch_name'   => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only contain {param} characters in maximum.',
            'min_length'    => '{field} must be {param} characters in minimum.',
        ]
    ];
    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = [];
    protected $afterInsert    = ['logInsert'];
    protected $beforeUpdate   = [];
    protected $afterUpdate    = ['logUpdate'];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = [];
    protected $afterDelete    = ['logDelete'];

    // Flags
    protected $skipLogAction  = false;

    /**
     * Activates the branch.
     * 
     * @param int|string $id
     */
    public function activate($id)
    {
        if (!$this->isActive($id)) {
            return $this->update($id, ['active' => true]);
        }

        return true;
    }

    /**
     * Activates multiple branch.
     * 
     * @param array|int|string $id
     */
    public function activateMultiple($id)
    {
        if (is_array($id)) {
            foreach ($id as $branchId) {
                $this->activate($branchId);
            }

            return true;
        }

        return $this->activate($id);
    }

    /**
     * Deactivates the branch.
     * 
     * @param int|string $id
     */
    public function deactivate($id)
    {
        if ($this->isActive($id)) {
            return $this->update($id, ['active' => false]);
        }

        return true;
    }

    /**
     * Deactivates the branch.
     * 
     * @param array|int|string $id
     */
    public function deactivateMultiple($id)
    {
        if (is_array($id)) {
            foreach ($id as $branchId) {
                $this->deactivate($branchId);
            }

            return true;
        }

        return $this->deactivate($id);
    }

    /**
     * Deletes a single or multiple branch.
     */
    public function deleteMultiple($id, $purge = false)
    {
        if (is_array($id)) {
            foreach ($id as $branchId) {
                $this->delete($branchId, $purge);
            }

            return true;
        }

        return $this->delete($id, $purge);
    }

    /**
     * Checks if branch is active.
     * 
     * @param int|string $id
     */
    public function isActive($id)
    {
        try {
            return $this->find($id)->active;
        } catch (\Exception $ex) {
            return false;
        }
    }

    /** A callback function for ensuring that branch will be deactivated on delete if state is active */
    protected function deactivateOnDelete(array $data)
    {
        if (isset($data['id'])) {

            $id = $data['id'];

            $this->skipLog(function () use ($id) {
                $this->update($id, ['active' => false]);
            });
        }

        return $data;
    }

    // Log callback methods
    protected function logInsert(array $data)
    {
        $this->logAction('insert', $data['id']);
        return $data;
    }

    protected function logUpdate(array $data)
    {
        $this->logAction('update', $data['id']);
        return $data;
    }

    protected function logDelete(array $data)
    {
        $this->logAction('delete', $data['id']);
        return $data;
    }

    protected function skipLog(callable $callback)
    {
        // Update skipLogAction flag first
        $this->skipLogAction = true;

        try {
            return $callback();
        } finally {
            // Reset to original state
            $this->skipLogAction = false;
        }
    }

    protected function logAction($action, $entityId)
    {
        if ($this->skipLogAction) {
            return;
        }

        $logsModel = new LogModel();
        $logsModel->logAction($action, $entityId, $this->table);
    }
}
