<?php

namespace App\Models;

use CodeIgniter\Model;

class EventModel extends Model
{
    protected $table            = 'events';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'object';
    protected $useSoftDeletes   = (ENVIRONMENT === 'production');
    protected $protectFields    = true;
    protected $allowedFields    = ['event_code', 'event_name', 'event_description', 'active', 'created_by'];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules      = [
        'id'            => [
            'label'         => 'ID',
            'rules'         => 'permit_empty',
        ],
        'event_code'    => [
            'label'         => 'Event Code',
            'rules'         => 'required|is_unique[events.event_code,id,{id}]|max_length[20]|alpha_dash',
        ],
        'event_name'    => [
            'label'         => 'Event Name',
            'rules'         => 'required|max_length[50]',
        ],
        'event_description' => [
            'label'         => 'Description',
            'rules'         => 'permit_empty|max_length[255]',
        ],
    ];
    protected $validationMessages   = [
        'event_code'    => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only contain {param} characters in maximum.',
            'is_unique'     => '{field} {value} already exists.',
            'regex_match'   => '{field} must only contain alphanumeric characters and the following symbols: . _ ~',
        ],
        'event_name'    => [
            'required'      => '{field} is required.',
            'max_length'    => '{field} must only contain {param} characters in maximum.',
        ],
        'event_description' => [
            'max_length'    => '{field} must only contain {param} characters in maximum.',
        ],
    ];
    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = ['ensureSingleActiveEvent'];
    protected $afterInsert    = ['logInsert'];
    protected $beforeUpdate   = ['ensureSingleActiveEvent'];
    protected $afterUpdate    = ['logUpdate'];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = ['deactivateOnDelete'];
    protected $afterDelete    = ['logDelete'];

    // Flags
    protected $skipLogAction  = false;

    /**
     * Activates the event.
     * 
     * @param int|string $id
     */
    public function activate($id): bool
    {
        if (!$this->isActive($id)) {
            return $this->update($id, ['active' => true]);
        }

        return true;
    }

    /**
     * Activates multiple event.
     * 
     * @param array|int|string $id
     */
    public function activateMultiple($id)
    {
        if (is_array($id)) {
            foreach ($id as $eventId) {
                $this->activate($eventId);
            }

            return true;
        }

        return $this->activate($id);
    }

    /**
     * Deactivates the event.
     * 
     * @param int|string $id
     */
    public function deactivate($id): bool
    {
        if ($this->isActive($id)) {
            return $this->update($id, ['active' => false]);
        }

        return true;
    }

    /**
     * Deactivates multiple event.
     * 
     * @param array|int|string $id
     */
    public function deactivateMultiple($id)
    {
        if (is_array($id)) {
            foreach ($id as $eventId) {
                $this->deactivate($eventId);
            }

            return true;
        }

        return $this->deactivate($id);
    }

    /**
     * Checks if the event is active.
     * 
     * @param int|string $id
     */
    public function isActive($id): bool
    {
        try {
            return $this->find($id)->active;
        } catch (\Exception $ex) {
            return false;
        }
    }

    /**
     * Deletes a single or multiple event.
     * 
     * @param array|int|string $id
     */
    public function deleteMultiple($id, $purge = false)
    {
        if (is_array($id)) {
            foreach ($id as $eventId) {
                $this->delete($eventId, $purge);
            }

            return true;
        }

        return $this->delete($id, $purge);
    }

    /** Returns the row of an active event (null if none) */
    public function findActiveEvent()
    {
        return $this->where(['active' => true])->first();
    }

    /** A callback function for ensuring only single event can be active */
    protected function ensureSingleActiveEvent(array $data)
    {
        if (isset($data['data']['active']) && $data['data']['active'] == true) {

            $this->skipLog(function () {
                $this->where('active', true)->set(['active' => false])->update();
            });
        }

        return $data;
    }

    /** A callback function for ensuring that event will be deactivated on delete if state is active */
    protected function deactivateOnDelete(array $data)
    {
        if (isset($data['id'])) {

            $id = $data['id'];

            $this->skipLog(function () use ($id) {
                $this->update($id, ['active' => false]);
            });
        }

        return $data;
    }

    // Log callback methods
    protected function logInsert(array $data)
    {
        $this->logAction('insert', $data['id']);

        return $data;
    }

    protected function logUpdate(array $data)
    {
        $this->logAction('update', $data['id']);

        return $data;
    }

    protected function logDelete(array $data)
    {
        $this->logAction('delete', $data['id']);

        return $data;
    }

    protected function skipLog(callable $callback)
    {
        // Update skipLogAction flag first
        $this->skipLogAction = true;

        try {
            return $callback();
        } finally {
            // Reset to original state
            $this->skipLogAction = false;
        }
    }

    protected function logAction($action, $entityId)
    {
        if ($this->skipLogAction) {
            return;
        }

        $logsModel = new LogModel();
        $logsModel->logAction($action, $entityId, $this->table);
    }
}
