<?php

declare(strict_types=1);

namespace App\Models;

use CodeIgniter\Shield\Models\UserModel as ShieldUserModel;

class UserModel extends ShieldUserModel
{
    protected $useSoftDeletes = false;

    // Flags
    protected $skipLogAction  = false;

    protected function initialize(): void
    {
        parent::initialize();

        $this->allowedFields = [
            ...$this->allowedFields,

            "branch_id",
            "first_name",
            "last_name",
            "middle_name",
            "phone_number",
            "created_by",
        ];

        $this->afterInsert = [
            ...$this->afterInsert,

            "logInsert"
        ];

        $this->afterUpdate = [
            ...$this->afterUpdate,

            "logUpdate"
        ];

        $this->afterDelete = [
            ...$this->afterDelete,

            "logDelete"
        ];
    }

    /**
     * Activates the user.
     */
    public function activateMultiple($id): bool
    {
        if (is_array($id)) {
            // return $this->whereIn('id', $id)->set('active', true)->update();
            $error = [];
            foreach ($id as $userid) {
                $user = $this->findById($userid);
                if ($user && $user->isNotActivated()) {
                    if (!$user->activate()) {
                        $error[] = $userid;
                    }
                }
            }

            return empty($error);
        }

        if ($user = $this->findById($id)) {
            return $user->activate();
        }

        return false;
    }

    /**
     * Deactivates the user.
     */
    public function deactivateMultiple($id): bool
    {
        if (is_array($id)) {
            // return $this->whereIn('id', $id)->set('active', false)->update();
            $error = [];
            foreach ($id as $userid) {
                $user = $this->findById($userid);
                if ($user && $user->isActivated()) {
                    if (!$user->deactivate()) {
                        $error[] = $userid;
                    }
                }
            }

            return empty($error);
        }

        if ($user = $this->findById($id)) {
            return $user->deactivate();
        }

        return false;
    }

    /**
     * Ban the user.
     */
    public function banMultiple($id): bool
    {
        if (is_array($id)) {
            // return $this->whereIn('id', $id)->set('status', 'banned')->update();
            $error = [];
            foreach ($id as $userid) {
                $user = $this->findById($userid);
                if ($user && $user->isBanned() === false) {
                    if ($user->ban()->isBanned() === false) {
                        $error[] = $userid;
                    }
                }
            }

            return empty($error);
        }

        if ($user = $this->findById($id)) {
            return $user->ban()->isBanned() === true;
        }

        return false;
    }

    /**
     * Unban the user.
     */
    public function unbanMultiple($id): bool
    {
        if (is_array($id)) {
            // return $this->whereIn('id', $id)->set('status', null)->update();
            $error = [];
            foreach ($id as $userid) {
                $user = $this->findById($userid);
                if ($user && $user->isBanned() === true) {
                    if ($user->unBan()->isBanned() === true) {
                        $error[] = $userid;
                    }
                }
            }

            return empty($error);
        }

        if ($user = $this->findById($id)) {
            return $user->unBan()->isBanned() === false;
        }

        return false;
    }

    /**
     * Deletes a single or multiple user.
     */
    public function deleteMultiple($id)
    {
        if (is_array($id)) {
            $error = [];
            foreach ($id as $userid) {
                if (!$this->delete($userid)) {
                    $error[] = $userid;
                }
            }

            return empty($error);
        }

        return $this->delete($id);
    }

    // Log callback methods
    protected function logInsert(array $data)
    {
        $this->logAction('insert', $data['id']);
        return $data;
    }

    protected function logUpdate(array $data)
    {
        $this->logAction('update', $data['id']);
        return $data;
    }

    protected function logDelete(array $data)
    {
        $this->logAction('delete', $data['id']);
        return $data;
    }

    protected function skipLog(callable $callback)
    {
        // Update skipLogAction flag first
        $this->skipLogAction = true;

        try {
            return $callback();
        } finally {
            // Reset to original state
            $this->skipLogAction = false;
        }
    }

    protected function logAction($action, $entityId)
    {
        if ($this->skipLogAction) {
            return;
        }

        $logsModel = new LogModel();
        $logsModel->logAction($action, $entityId, $this->table);
    }
}
